// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


package com.vasco.orchestration.sample.utils;

import static androidx.biometric.BiometricManager.Authenticators.BIOMETRIC_STRONG;
import static androidx.biometric.BiometricManager.Authenticators.DEVICE_CREDENTIAL;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.util.Log;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.fragment.app.FragmentActivity;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;
import androidx.biometric.BiometricPrompt;
import androidx.core.content.ContextCompat;

import com.vasco.orchestration.client.authentication.UserAuthenticationCallback;
import com.vasco.orchestration.client.authentication.UserAuthenticationInputCallback;
import com.vasco.orchestration.client.errors.InputError;
import com.vasco.orchestration.client.errors.OrchestrationError;
import com.vasco.orchestration.client.errors.OrchestrationErrorCallback;
import com.vasco.orchestration.client.errors.OrchestrationServerError;
import com.vasco.orchestration.client.errors.OrchestrationWarning;
import com.vasco.orchestration.client.errors.OrchestrationWarningCallback;
import com.vasco.orchestration.client.exceptions.AuthenticationInputException;
import com.vasco.orchestration.sample.R;
import com.vasco.orchestration.sample.UserAuthenticationDialogFragment;

import java.util.Objects;
import java.util.concurrent.Executor;

public class SampleOrchestrationCallback
        implements OrchestrationErrorCallback,
                OrchestrationWarningCallback,
                UserAuthenticationCallback {

    private final Context context;
    private Dialog progressDialog;
    private UserAuthenticationInputCallback inputCallback;
    private static final String DIALOG_TAG = "dialog";


    public SampleOrchestrationCallback(Context context) {
        this.context = context;
    }

    @Override
    public void onOrchestrationError(OrchestrationError error) {
        // Hide progress & display error
        UIUtils.hideProgress(progressDialog);
        UIUtils.displayAlert(
                context,
                context.getString(R.string.dialog_error_title),
                ErrorUtils.getErrorMessage(context, error.getErrorCode()));
        Log.e(
                context.getClass().getName(),
                "Exception in onOrchestrationError",
                error.getException());
    }

    @Override
    public void onOrchestrationServerError(OrchestrationServerError error) {
        // Hide progress & display readable message
        UIUtils.hideProgress(progressDialog);
        UIUtils.displayAlert(
                context,
                context.getString(R.string.dialog_error_title),
                error.getReadableMessage());
        Log.e(
                context.getClass().getName(),
                "Payload in onOrchestrationServerError: " + error.getCustomPayload());
    }

    @Override
    public void onOrchestrationWarning(OrchestrationWarning warning) {
        String warningMessage = ErrorUtils.getWarningMessage(context, warning.getWarningCode());
        Log.w(context.getClass().getName(), warningMessage);
    }

    public void setProgressDialog(Dialog progressDialog) {
        this.progressDialog = progressDialog;
    }

    // UserAuthenticationCallback
    @Override
    public void onUserAuthenticationRequired(
            UserAuthentication type,
            final UserAuthenticationInputCallback inputCallback,
            final boolean isEnrollment)
    {
        this.inputCallback = inputCallback;

        if (type == UserAuthentication.PASSWORD) {
            startPinPadAuthentication(inputCallback, isEnrollment);
        } else if (type == UserAuthentication.BIOMETRIC) {
            startBiometricAuthentication(inputCallback);
        } else if(type == UserAuthentication.PRESENCE) {
            startPresenceAuthentication(inputCallback);
        }
    }

    private void startPinPadAuthentication(UserAuthenticationInputCallback inputCallback, boolean isEnrollment) {
        UserAuthenticationDialogFragment dialog =
                new UserAuthenticationDialogFragment(
                        (Activity) context, inputCallback, isEnrollment, 0);
        FragmentManager fragmentManager = ((FragmentActivity) context).getSupportFragmentManager();
        FragmentTransaction ft = fragmentManager.beginTransaction();
        ft.add(dialog, DIALOG_TAG);
        ft.commitAllowingStateLoss();
    }

    private void startBiometricAuthentication(UserAuthenticationInputCallback inputCallback) {
        BiometricPrompt.PromptInfo promptInfo =
                new BiometricPrompt.PromptInfo.Builder()
                        .setTitle(context.getString(R.string.external_biometric_authentication))
                        .setSubtitle(context.getString(R.string.authenticate_user))
                        .setAllowedAuthenticators(BIOMETRIC_STRONG | DEVICE_CREDENTIAL)
                        .build();
        Executor executor = ContextCompat.getMainExecutor(context);
        BiometricPrompt biometricPrompt =
                new BiometricPrompt(
                        (FragmentActivity) context,
                        executor,
                        new ExternalUserAuthenticationCallback(context, inputCallback));
        biometricPrompt.authenticate(promptInfo);
    }

    private void startPresenceAuthentication(UserAuthenticationInputCallback inputCallback) {
        PresenceAuthenticationDialogFragment dialog =
                new PresenceAuthenticationDialogFragment((Activity) context, inputCallback);
        FragmentManager fragmentManager = ((FragmentActivity) context).getSupportFragmentManager();
        FragmentTransaction ft = fragmentManager.beginTransaction();
        ft.add(dialog, DIALOG_TAG);
        ft.commitAllowingStateLoss();
    }


    @Override
    public void onUserAuthenticationInputError(InputError error) {
        UserAuthenticationDialogFragment dialog =
                new UserAuthenticationDialogFragment(
                        (Activity) context, inputCallback, true, R.string.orch_pinpad_error_weak);
        FragmentManager fragmentManager = ((FragmentActivity) context).getSupportFragmentManager();
        dialog.show(fragmentManager, DIALOG_TAG);
    }
}

class ExternalUserAuthenticationCallback extends BiometricPrompt.AuthenticationCallback {
    private final UserAuthenticationInputCallback inputCallback;
    private final Context context;

    ExternalUserAuthenticationCallback(
            Context context,
            UserAuthenticationInputCallback inputCallback
    ){
        this.inputCallback = inputCallback;
        this.context = context;
    }

    @Override
    public void onAuthenticationSucceeded(
            @NonNull BiometricPrompt.AuthenticationResult result
    ) {
        super.onAuthenticationSucceeded(result);
        try {
            inputCallback.onUserAuthenticationSuccess();
        } catch (AuthenticationInputException e) {
            Toast.makeText(context, e.getMessage(), Toast.LENGTH_SHORT).show();
            Log.e(context.getClass().getName(), Objects.requireNonNull(e.getMessage()));
            inputCallback.onUserAuthenticationAborted();
        }
    }
}
